/******************************************************************************
*                                                                             *
* Freescale Semiconductor Inc.                                                *
* (c) Copyright 2004-2008 Freescale Semiconductor, Inc.                       *
* ALL RIGHTS RESERVED.                                                        *
*                                                                             *
*******************************************************************************
*******************************************************************************
* Example for S08QG8  - I2C Master + SCI to PC;                               *
*                       MCU acts as MCU of STB;                               *
******************************************************************************/

#include <hidef.h>          /* for EnableInterrupts macro */
#include "derivative.h"     /* include peripheral declarations */
#include "application.h"    /* Application definitions */


//*****************************************************************************
void main(void) 
{
    MCU_init(); /* call Device Initialization */
    Init_Variables();
    Test_IIC_Bus();
    SCI_send_byte(0x0D);
    SCI_send_byte(0x0A);
    SCI_send_string(PWR_ON);
    
    for(;;) 
    {
        __RESET_WATCHDOG();
        
        if(flag)
        {
            switch (RD_data) 
            {
                case 0x53:      // "S";
                case 0x73:      // "s";
                case ' ':       // space;
                {
                    SCI_send_string(PWR_OFF);
                    IIC_write_byte(STB_OFF);
                    flag = 0;
                }
                break;
            }
        }
    }
}
/********************************************************************************
*                                                                               *
*   Description: Initialize MCU registers                                       *
*                                                                               *
********************************************************************************/
void MCU_init(void)
{
    SOPT1 = 0xD2;
    SPMSC1 = 0x1C; 
    SPMSC2 = 0x00;
    SPMSC3 &= (unsigned char)~0x30;
    ICSTRM = *(unsigned char*far)0xFFAF;
    ICSSC = *(unsigned char*far)0xFFAE;
    ICSC1 = 0x04; 
    ICSC2 = 0x00;
    SOPT2 = 0x00;
    /* Common initialization of the CPU registers */
    PTASE |= (unsigned char)0x1F;
    PTAPE_PTAPE1 = 1;   // pull-up enabled;
    PTAPE_PTAPE2 = 1;   // pull-up enabled;
    PTAPE_PTAPE3 = 1;   // pull-up enabled;
    PTADD = 0;
    PTBSE = 0xFF;
    PTADS = 0x00;
    PTBDS = 0x00;
    /* ### Init_SCI code */
    SCIC2 = 0x00;
    (void)(SCIS1 == 0);
    (void)(SCID == 0);
    SCIS2 = 0x00;
    SCIBD = 0x34;
    SCIC1 = 0x04;
    SCIC2 = 0x2C;
    SCIC3 = 0x20;
    /* ### Init_IIC code */
    IICF = 0x97;        // 0x87 => 50kHz; 0x47 => 100kHz; 0x4C => 86kHz;
    IICA = 0x00;
    IICS = 0x12;
    IICC |= (unsigned char)0x80;
    IICC = 0x98;
    SRS = 0xFF;
    asm CLI;
}
/********************************************************************************
*   Description: Reads data byte by I2C bus;                                    *
*                Defined I2C Slave address used;                                *
*                                                                               *
*   Input:  NO;                                                                 *
*                                                                               *
*   Output: byte variable RD_data;                                              *
*                                                                               *
********************************************************************************/
byte IIC_read_byte(void)
{
    IICC_TXAK = 0;          // RX/TX = 1; MS/SL = 1; TXAK = 0;
    IICC |= 0x30;           // And generate START condition;

    IICD = IIC_SLAVE;       // Adress the slave and set up for master transmit;
    while (!IICS_IICIF);    // wait until IBIF;
    IICS_IICIF=1;           // clear the interrupt event flag;
    while(IICS_RXAK);       // check for RXAK;
        //-----Slave ACK occured------------     
    IICC_RSTA = 1;          // set up repeated start;

    IICD = IIC_SLAVE | 1;       // (slave_address) | (RW = 1);
    while (!IICS_IICIF);    // wait until IBIF;
    IICS_IICIF=1;           // clear the interrupt event flag;
    while (IICS_RXAK);      // check for RXAK;
        //-----Slave ACK occured------------  
    IICC_TX = 0;            // set up to receive;
    IICC_TXAK  = 1;         // acknowledge disable; 
    RD_data = IICD;         // dummy read;
    while (!IICS_IICIF);    // wait until IBIF;
    IICS_IICIF=1;           // clear the interrupt event flag;

    IICC_MST = 0;           // generate stop signal;

    RD_data = IICD;         // read right data;
    return RD_data;
}
/********************************************************************************
*   Description: Send data byte to I2C bus;                                     *
*                Defined I2C Slave address used;                                *
*                                                                               *
*   Input:  byte variable data;                                                 *
*                                                                               *
*   Output: NO;                                                                 *
*                                                                               *
********************************************************************************/
void IIC_write_byte(byte data)
{
    IICC_TXAK = 0;         // RX/TX = 1; MS/SL = 1; TXAK = 0;
    IICC |= 0x30;          // And generate START condition;
        //-------start of transmit first byte to IIC bus-----
    IICD = IIC_SLAVE;      // Adress the slave and set up for master transmit;
    while (!IICS_IICIF);	 // wait until IBIF;
    IICS_IICIF=1;          // clear the interrupt event flag;
    while(IICS_RXAK);      // check for RXAK;
        //-----Slave ACK occured------------
    IICD = data;
    while (!IICS_IICIF);	 // wait until IBIF;
    IICS_IICIF=1;          // clear the interrupt event flag;
    while(IICS_RXAK);      // check for RXAK;
        //-----Slave ACK occured------------
    IICS_IICIF=1;          // clear the interrupt event flag;
    IICC_MST = 0;          // generate STOP condition;
}
/********************************************************************************
*   Description: Test whether I2C bus is in BUSY state;                         *
*                                                                               *
*   Input:  NO;                                                                 *
*                                                                               *
*   Output: MCU check the flags for 15msec and makes RESET I2C if need;         *
*                                                                               *
********************************************************************************/
void Test_IIC_Bus(void)
{
    if((IICS_BUSY) || (IICS_ARBL)) WaitNms(5);
    if((IICS_BUSY) || (IICS_ARBL)) WaitNms(5);
    if((IICS_BUSY) || (IICS_ARBL)) WaitNms(5);
    if((IICS_BUSY) || (IICS_ARBL)) Reset_IIC();
}
/********************************************************************************
*   Description: Reset I2C bus;                                                 *
*                                                                               *
*   Input:  NO;                                                                 *
*                                                                               *
*   Output: MCU send 9 SCL pulses to maintain RESET of the I2C device connected;*
*                                                                               *
********************************************************************************/
void Reset_IIC(void)
{
    byte i;

    IICC_IICEN = 0;    // Disable IIC;
    PTADD_PTADD2 = 1;
    PTADD_PTADD3 = 1;
    SCL = 1;
    Delay(10);
    SDA = 0;
    Delay(10);
    for(i=0;i<10;i++)
    {
        SCL = 1;
        Delay(10);
        SCL = 0;
        Delay(10);
    }
    SCL = 1;
    Delay(10);
    SDA = 1;
    PTADD_PTADD2 = 0;
    PTADD_PTADD3 = 0;
    IICC_IICEN = 1;    // Enable IIC;
    Delay(20);
} 
/********************************************************************************
*   Description: send byte to SCI;                                              *
*                                                                               *
*   Input:  byte data;                                                          *
*                                                                               *
*   Output: NO;                                                                 *
*                                                                               *
********************************************************************************/
void SCI_send_byte(byte data)
{
    while(!SCIS1_TC);     // wait for transmit buffer empty;
    SCID = data;          // send data;
}
/********************************************************************************
*   Description: send string to SCI;                                            *
*                                                                               *
*   Input:  pointer to string;                                                  *
*                                                                               *
*   Output: NO;                                                                 *
*                                                                               *
********************************************************************************/
void SCI_send_string(const byte *str)
{
    byte m;
    for (m=0;m<LENGTH;m++)
	{	
	    SCI_send_byte(str[m]); // write the character to the SCI interface;
	}
	SCI_send_byte(0x0D);
    SCI_send_byte(0x0A);	 
}
/********************************************************************************
*   Description: set initial values of variebles;                               *
*                                                                               *
*   Input:  NO;                                                                 *
*                                                                               *
*   Output: default values for variables;                                       *
*                                                                               *
********************************************************************************/
void Init_Variables(void)
{
    WR_data = 0;
    RD_data = 0;
    my_IIC_RX_data = 0;
    my_IIC_TX_data = 0;
    IIC_Res_flg = 0;
    flag = 0;
}
/********************************************************************************
*   Time delay functions                                                        *
********************************************************************************/
void Delay(byte del)                    // Universal short delay;
{
    byte i;
    for(i=0;i<del;i++) __RESET_WATCHDOG();
}
//-----------------------------------------------------------------------------
void Wait1ms(void)                      // for defined BusCLK;
{
    asm {  
        PSHX
        PSHH
        LDHX #CONST
     loop:
        AIX #-1  ;2 cycles
        CPHX #0  ;3 cycles
        BNE loop ;3 cycles              
        PULH
        PULX
       }
    __RESET_WATCHDOG(); 
}
//-----------------------------------------------------------------------------
void WaitNms(word del)                  // for defined BusCLK;
{
    word i;
    for(i=0;i<del;i++) Wait1ms(); 
}

/********************************************************************************
*   Description: SCI RX interrupt service routine;                              *
*                                                                               *
*   Input:  SCI;                                                                *
*                                                                               *
*   Output: byte variable RD_data,  flag = 1;                                   *
*                                                                               *
********************************************************************************/
__interrupt IV(Vscirx) void isrVscirx(void)
{
    temp = SCIS1; 
    RD_data = SCID;
    flag = 1;
}     

